<?php
/**
 * @author WP Cloud Plugins
 * @copyright Copyright (c) 2022, WP Cloud Plugins
 *
 * @since       2.0
 * @see https://www.wpcloudplugins.com
 */

namespace TheLion\UseyourDrive;

class Download
{
    /**
     * @var \TheLion\UseyourDrive\CacheNode
     */
    public $cached_node;

    /**
     * @var string 'proxy'|'redirect'
     */
    private $_download_method;

    /**
     * Should the file be streamed using the server as middleman.
     *
     * @var bool
     */
    private $_force_proxy;

    /**
     * Url to the content of the file. Is only public when file is shared.
     *
     * @var string
     */
    private $_content_url;

    /**
     * Mimetype of the download.
     *
     * @var string
     */
    private $_mimetype;

    /**
     * Extension of the file.
     *
     * @var string
     */
    private $_extension;

    /**
     * Is the download streamed (for media files).
     *
     * @var bool
     */
    private $_is_stream = false;

    public function __construct(CacheNode $cached_node, $mimetype = 'default', $force_proxy = false)
    {
        $this->cached_node = $cached_node;

        // Use the orginial entry if the file/folder is a shortcut
        if ($cached_node->is_shortcut()) {
            $original_node = $cached_node->get_original_node();

            if (!empty($original_node)) {
                $this->cached_node = $original_node;
            }
        }

        $this->_force_proxy = $force_proxy;

        $this->_mimetype = $_REQUEST['mimetype'] ?? $mimetype;
        $this->_extension = $_REQUEST['extension'] ?? $this->get_entry()->get_extension();
        $this->_is_stream = (isset($_REQUEST['action']) && 'useyourdrive-stream' === $_REQUEST['action']);

        // Update mimetype for Google Docs
        $export_formats = $this->cached_node->get_entry()->get_save_as();

        if ('default' === $this->_mimetype && (count($export_formats) > 0)) {
            $format = $export_formats['PDF'] ?? reset($export_formats);
            $this->_mimetype = $format['mimetype'];
            $this->_extension = $format['extension'];
        }

        $this->_set_content_url();

        wp_using_ext_object_cache(false);

        $this->_set_download_method();
    }

    public function start_download()
    {
        // Execute download Hook
        do_action('useyourdrive_download', $this->get_cached_node(), $this);

        // Log Download
        if ('default' === $this->_mimetype) {
            $event_type = $this->is_stream() ? 'useyourdrive_streamed_entry' : 'useyourdrive_downloaded_entry';

            if ('useyourdrive_streamed_entry' === $event_type && in_array($this->get_entry()->get_extension(), ['vtt', 'srt'])) {
                // Don't log VTT captions when requested for video stream
            } else {
                do_action('useyourdrive_log_event', $event_type, $this->get_cached_node());
            }
        } else {
            do_action('useyourdrive_log_event', 'useyourdrive_downloaded_entry', $this->get_cached_node(), ['exported' => strtoupper($this->get_extension())]);
        }

        // Send email if needed
        if ('1' === Processor::instance()->get_shortcode_option('notificationdownload') && !$this->is_stream()) {
            Processor::instance()->send_notification_email('download', [$this->get_cached_node()]);
        }

        // Finally, start the download
        $this->_process_download();
    }

    public function redirect_to_content()
    {
        // Get the download link via the webContentLink
        header('Location: '.$this->get_content_url());

        exit;
    }

    public function redirect_to_export()
    {
        header('Location: '.$this->get_content_url());

        exit;
    }

    public function stream_content()
    {
        if (function_exists('apache_setenv')) {
            @apache_setenv('no-gzip', 1);
        }
        @ini_set('zlib.output_compression', 'Off');
        @session_write_close();

        // Stop WP from buffering
        wp_ob_end_flush_all();

        $chunk_size = min(Helpers::get_free_memory_available() - (1024 * 1024 * 5), 1024 * 1024 * 50); // Chunks of 50MB or less if memory isn't sufficient

        $size = $this->get_cached_node()->get_entry()->get_size();

        $length = $size;           // Content length
        $start = 0;               // Start byte
        $end = $size - 1;       // End byte
        header('Accept-Ranges: bytes');
        header('Content-Type: '.$this->get_cached_node()->get_entry()->get_mimetype());

        $seconds_to_cache = 60 * 60 * 24;
        $ts = gmdate('D, d M Y H:i:s', time() + $seconds_to_cache).' GMT';
        header("Expires: {$ts}");
        header('Pragma: cache');
        header("Cache-Control: max-age={$seconds_to_cache}");

        if (isset($_SERVER['HTTP_RANGE'])) {
            $c_start = $start;
            $c_end = $end;
            list(, $range) = explode('=', $_SERVER['HTTP_RANGE'], 2);

            if (false !== strpos($range, ',')) {
                header('HTTP/1.1 416 Requested Range Not Satisfiable');
                header("Content-Range: bytes {$start}-{$end}/{$size}");

                exit;
            }

            if ('-' == $range) {
                $c_start = $size - substr($range, 1);
            } else {
                $range = explode('-', $range);
                $c_start = (int) $range[0];

                if (isset($range[1]) && is_numeric($range[1])) {
                    $c_end = (int) $range[1];
                } else {
                    $c_end = $size;
                }

                if ($c_end - $c_start > $chunk_size) {
                    $c_end = $c_start + $chunk_size;
                }
            }
            $c_end = ($c_end > $end) ? $end : $c_end;

            if ($c_start > $c_end || $c_start > $size - 1 || $c_end >= $size) {
                header('HTTP/1.1 416 Requested Range Not Satisfiable');
                header("Content-Range: bytes {$start}-{$end}/{$size}");

                exit;
            }

            $start = $c_start;

            $end = $c_end;
            $length = $end - $start + 1;
            header('HTTP/1.1 206 Partial Content');
        }

        header("Content-Range: bytes {$start}-{$end}/{$size}");
        header('Content-Length: '.$length);

        $chunk_start = $start;

        @set_time_limit(0);

        while ($chunk_start <= $end) {
            // Output the chunk

            $chunk_end = ((($chunk_start + $chunk_size) > $end) ? $end : $chunk_start + $chunk_size);
            $this->_stream_get_chunk($chunk_start, $chunk_end);

            $chunk_start = $chunk_end + 1;
        }
    }

    /**
     * Callback function for CURLOPT_WRITEFUNCTION, This is what prints the chunk.
     *
     * @param CurlHandle $ch
     * @param string     $str
     *
     * @return type
     */
    public function _stream_chunk_to_output($ch, $str)
    {
        echo $str;

        return strlen($str);
    }

    /**
     * Exports are generated on the fly via the API, so can't download this in chunks.
     */
    public function export_content()
    {
        // Stop WP from buffering
        wp_ob_end_flush_all();

        $export_link = $this->get_entry()->get_export_link($this->_mimetype);

        if (($this->get_entry()->get_size() <= 10485760) && (empty($export_link) || false === API::has_permission($this->get_cached_node()->get_id()) || 'proxy' === $this->get_download_method())) {
            // Only use export link if publicly accessible and the exported document is smaller than 10MB (API Limit 'exportSizeLimitExceeded')
            $export_link = $this->get_api_url();
        } else {
            header('Location: '.$export_link);

            return;
        }

        $request = new \UYDGoogle_Http_Request($export_link, 'GET');
        $httpRequest = App::instance()->get_sdk_client()->getAuth()->authenticatedRequest($request);
        $headers = $httpRequest->getResponseHeaders();

        if (isset($headers['location'])) {
            header('Location: '.$headers['location']);
        } else {
            foreach ($headers as $key => $header) {
                if ('transfer-encoding' === $key) {
                    continue;
                }

                if (is_array($header)) {
                    header("{$key}: ".implode(' ', $header));
                } else {
                    header("{$key}: ".str_replace("\n", ' ', $header));
                }
            }
        }

        echo $httpRequest->getResponseBody();
    }

    public function get_api_url()
    {
        if ('default' !== $this->_mimetype) {
            return 'https://www.googleapis.com/drive/v3/files/'.$this->get_cached_node()->get_id().'/export?alt=media&mimeType='.$this->_mimetype;
        }

        return 'https://www.googleapis.com/drive/v3/files/'.$this->get_cached_node()->get_id().'?alt=media';
    }

    public function get_content_url()
    {
        return $this->_content_url;
    }

    public function get_download_method()
    {
        return $this->_download_method;
    }

    public function get_cached_node()
    {
        return $this->cached_node;
    }

    public function get_entry()
    {
        return $this->get_cached_node()->get_entry();
    }

    public function get_mimetype()
    {
        return $this->_mimetype;
    }

    public function get_extension()
    {
        return $this->_extension;
    }

    public function is_stream()
    {
        return $this->_is_stream;
    }

    public function get_force_proxy()
    {
        return $this->_force_proxy;
    }

    public function set_force_proxy($_force_proxy)
    {
        $this->_force_proxy = $_force_proxy;
    }

    private function _set_content_url()
    {
        $direct_download_link = $this->get_entry()->get_direct_download_link();

        // Set download URL for binary files
        if ('default' === $this->get_mimetype() && !empty($direct_download_link)) {
            return $this->_content_url = $direct_download_link.'&userIp='.Helpers::get_user_ip();
        }

        // Set download URL for exporting documents with specific mimetype
        return $this->_content_url = $this->get_entry()->get_export_link($this->_mimetype).'&userIp='.Helpers::get_user_ip();
    }

    /**
     * Set the download method for this entry
     * Files can be streamed using the server as a proxy ('proxy') or
     * the user can be redirected to download url ('redirect').
     *
     * As the Google API doesn't offer temporarily download links,
     * the specific download method depends on several settings
     *
     * @return 'proxy'|'redirect'
     */
    private function _set_download_method()
    {
        // Is plugin forced to use the proxy method via the plugin options?
        if ($this->_force_proxy || 'proxy' === Processor::instance()->get_setting('download_method')) {
            return $this->_download_method = 'proxy';
        }

        // Files larger than 25MB can only be streamed unfortunately
        // There isn't a direcly download link available for those files and
        // a cookie security check by Google prevents them to be downloaded directly.
        if ($this->get_entry()->get_size() > 25165824) {
            return $this->_download_method = 'proxy';
        }

        // Is download via shared links prohibitted by API?
        $copy_disabled = $this->get_entry()->get_permission('copyRequiresWriterPermission');
        if ($copy_disabled) {
            return $this->_download_method = 'proxy';
        }

        // Is file already shared ?
        $is_shared = API::has_permission($this->get_cached_node()->get_id());
        if ($is_shared) {
            return $this->_download_method = 'redirect';
        }

        // Can the sharing permissions of the file be updated via the plugin?
        $can_update_permissions = ('Yes' === Processor::instance()->get_setting('manage_permissions')) && $this->get_entry()->get_permission('canshare');
        if (false === $can_update_permissions) {
            return $this->_download_method = 'proxy';
        }

        // Update the Sharing Permissions
        $is_sharing_permission_updated = API::set_permission($this->get_cached_node()->get_id());
        if (false === $is_sharing_permission_updated) {
            return $this->_download_method = 'proxy';
        }

        return $this->_download_method = 'redirect';
    }

    private function _process_download()
    {
        switch ($this->get_download_method()) {
            case 'proxy':
                if ('default' === $this->_mimetype) {
                    if (isset($_REQUEST['action']) && 'useyourdrive-stream' === $_REQUEST['action']) {
                        $this->stream_content();
                    } else {
                        $filename = $this->get_cached_node()->get_name();
                        header('Content-Disposition: attachment; '.sprintf('filename="%s"; ', rawurlencode($filename)).sprintf("filename*=utf-8''%s", rawurlencode($filename)));
                        $this->stream_content();
                    }
                } else {
                    $this->export_content();
                }

                break;

            case 'redirect':
                if ('default' === $this->_mimetype) {
                    $this->redirect_to_content();
                } else {
                    $this->export_content();
                }

                break;
        }

        exit;
    }

    /**
     * Start the stream, set all the Headers and return the size of stream.
     *
     * @param type  $url
     * @param mixed $download
     *
     * @return type
     */
    private function _stream_start($download = true)
    {
        $headers = [];

        // Add Resources key to give permission to access the item
        if ($this->get_entry()->has_resourcekey()) {
            $headers['X-Goog-Drive-Resource-Keys'] = $this->get_entry()->get_id().'/'.$this->get_entry()->get_resourcekey();
        }

        $request = new \UYDGoogle_Http_Request($this->get_api_url(), 'GET');
        $request->disableGzip();

        App::instance()->get_sdk_client()->getIo()->setOptions(
            [
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HEADER => true,
                CURLOPT_NOBODY => true,
                CURLOPT_RANGE => 0 .'-',
            ]
        );
        $httpRequest = App::instance()->get_sdk_client()->getAuth()->authenticatedRequest($request);

        $headers = $httpRequest->getResponseHeaders();

        foreach ($headers as $key => $header) {
            if ('transfer-encoding' === $key) {
                continue;
            }

            if ('content-length:' === $key && false === $download) {
                continue;
            }

            if ('content-range:' === $key && false === $download) {
                continue;
            }

            if (is_array($header)) {
                header("{$key}: ".implode(' ', $header));
            } else {
                header("{$key}: ".str_replace("\n", ' ', $header));
            }
        }

        if (isset($headers['content-length'])) {
            return (int) $headers['content-length'];
        }

        return -1;
    }

    /**
     * Function to get a range of bytes via the API.
     *
     * @param int $start
     * @param int $end
     */
    private function _stream_get_chunk($start, $end)
    {
        $headers = ['Range' => 'bytes='.$start.'-'.$end];

        // Add Resources key to give permission to access the item
        if ($this->get_entry()->has_resourcekey()) {
            $headers['X-Goog-Drive-Resource-Keys'] = $this->get_entry()->get_id().'/'.$this->get_entry()->get_resourcekey();
        }

        $request = new \UYDGoogle_Http_Request($this->get_api_url(), 'GET', $headers);
        $request->disableGzip();

        App::instance()->get_sdk_client()->getIo()->setOptions(
            [
                CURLOPT_RETURNTRANSFER => false,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_RANGE => null,
                CURLOPT_NOBODY => null,
                CURLOPT_HEADER => false,
                CURLOPT_WRITEFUNCTION => [$this, '_stream_chunk_to_output'],
                CURLOPT_CONNECTTIMEOUT => null,
                CURLOPT_TIMEOUT => null,
            ]
        );

        App::instance()->get_sdk_client()->getAuth()->authenticatedRequest($request);
    }
}
