<?php
defined('ABSPATH') || die('No direct script access allowed!');
/**
 * One Drive
 *
 *
 * @link       https://themeforest.net/user/nouthemes/portfolio
 * @since      1.0.0
 *
 * @package    Cloudfiles_Wordpress_Media_Library_Folders_Cloud
 * @subpackage Cloudfiles_Wordpress_Media_Library_Folders_Cloud/includes/providers
 * @author     Nouthemes <nguyenvanqui89@gmail.com>
 */
use GuzzleHttp\Client as GuzzleHttpClient;
use Krizalys\Onedrive\Client;
use Krizalys\Onedrive\Onedrive;
use Microsoft\Graph\Graph;

class Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Provider_Onedrive extends Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Provider_Base {

	private $client_id;
	private $client_secret;
	private $access_token;

	public function __construct($args=[])
	{
		$this->name = esc_html__( 'One Drive', 'cloudfiles-wordpress-media-library-folders-cloud' );
		$this->slug = CLOUDFILES_WORDPRESS_MEDIA_LIBRARY_FOLDERS_CLOUD_ONEDRIVE;
        $this->taxonomy = CLOUDFILES_WORDPRESS_MEDIA_LIBRARY_FOLDERS_CLOUD_ONEDRIVE_TANONOMY_NAME;

		$this->client_id = get_option('onedrive_client_id');
		$this->client_secret = get_option('onedrive_client_secret');
        $this->access_token = get_option('onedrive_access_token');
		
		parent::__construct($args);
	}

	public function is_setup(){
		$enable = get_option('noucloudfiles_enable_onedrive', 'yes');
		if($enable == 'no'){
			return false;
        }
        
		if( !empty($this->client_id) && !empty($this->client_secret) ){
            $onedriveconfig = get_option('_cloudfiles_onedrive_config');
            if (isset($onedriveconfig['state']) && isset($onedriveconfig['state']->token->data->access_token)) {
                return true;
            }else{
                if(isset($_GET['tab']) && $_GET['tab'] == $this->slug){
                    Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Settings::remove_error();
                    Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Settings::add_error( esc_html__('Please, authorize account.', 'cloudfiles-wordpress-media-library-folders-cloud') );
                }
            }
		}
		return false;
	}

    /**
     * Read OneDrive app key and secret
     *
     * @return Client|OneDrive_Client|boolean
     */
    public function getClient()
    {
        $onedriveconfig = get_option('_cloudfiles_onedrive_config');
        try {
            if (isset($onedriveconfig['state']) && isset($onedriveconfig['state']->token->data->access_token)) {
                $graph = new Graph();
                $graph->setAccessToken($onedriveconfig['state']->token->data->access_token);
                $client = new Client(
                    $this->client_id,
                    $graph,
                    new GuzzleHttpClient(),
                    array(
                        'state' => $onedriveconfig['state']
                    )
                );

                if ($client->getAccessTokenStatus() === -2) {
                    $client = $this->renewAccessToken($client, $onedriveconfig);
                }

                $this->client = $graph;
                return $this->client;
            }
        } catch (Exception $ex) {
            return new WP_Error('broke', esc_html__('Error communicating with OneDrive API: ') . $ex->getMessage());
        }

        return false;
    }
	
	public function init(){
        $this->client = $this->getClient();
        if ($this->client instanceof Graph) {
            $this->adapter = new Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Adapter_OneDrive($this->client);
		    $this->filesystem = new Cloudfiles_Wordpress_Media_Library_System_Filesystem($this->adapter, ['disable_asserts' => true]);
        }
	}

    /**
     * Get a $authorizeUrl
     *
     * @return string|WP_Error
     */
    public function getAuthUrl()
    {
    	$client = Onedrive::client($this->client_id);
        try {
            $onedriveconfig = get_option('_cloudfiles_onedrive_config');
            $authorizeUrl = $client->getLogInUrl(array(
                'files.read',
                'files.read.all',
                'files.readwrite',
                'files.readwrite.all',
                'offline_access',
            ), admin_url() );

            $onedriveconfig['state'] = $client->getState();
            update_option('_cloudfiles_onedrive_config', $onedriveconfig);
        } catch (Exception $ex) {
            return new WP_Error('broke', esc_html__('Could not start authorization: ') . $ex->getMessage());
        }
        return $authorizeUrl;
    }

    /**
     * Create token after connected
     *
     * @param string $code Code to access to onedrive app
     *
     * @return boolean|WP_Error
     */
    public function createToken($code)
    {
        try {
            $onedriveconfig = get_option('_cloudfiles_onedrive_config');
	    	$client = Onedrive::client(
			    $this->client_id,
			    ['state' => $onedriveconfig['state']]
			);
			$client->obtainAccessToken($this->client_secret, $code);
			$token = $client->getState()->token->data->access_token;
			$refresh_token = $client->getState()->token->data->refresh_token;
			$onedriveconfig['connected'] = 1;
	        $onedriveconfig['state'] = $client->getState();
	        update_option('_cloudfiles_onedrive_config', $onedriveconfig);
	        update_option('onedrive_access_token', $token);
	        update_option('onedrive_refresh_token', $refresh_token);

	        wp_redirect(admin_url('admin.php?page=cloudfiles_wordpress_media_library_folders_cloud&authorize=1&tab='.$this->slug));
			exit;

        } catch (Exception $ex) {
            wp_die( 
				esc_html__('Error communicating with OneDrive API: ') . $ex->getMessage(),
				'',
				[
					'link_url' => esc_url(admin_url('admin.php?page=cloudfiles_wordpress_media_library_folders_cloud')),
					'link_text' => esc_html__('Back to Cloudfiles', 'cloudfiles-wordpress-media-library-folders-cloud'),
				] 
			);
        }
    }

    /**
     * Renews the access token from OAuth. This token is valid for one hour.
     *
     * @param object $client         Client
     * @param array  $onedriveconfig Setings
     *
     * @return Client
     */
    public function renewAccessToken($client, $onedriveconfig)
    {
        $client->renewAccessToken($this->client_secret);
        $onedriveconfig['state'] = $client->getState();
        update_option('_cloudfiles_onedrive_config', $onedriveconfig);
        $graph = new Graph();

        $access_token = $client->getState()->token->data->access_token;
		$refresh_token = $client->getState()->token->data->refresh_token;

		update_option('onedrive_access_token', $access_token);
	    update_option('onedrive_refresh_token', $refresh_token);

        $graph->setAccessToken($access_token);
        $client = new Client(
            $this->client_id,
            $graph,
            new GuzzleHttpClient(),
            array(
                'state' => $client->getState()
            )
        );

        return $client;
    }

    /**
     * Get share link
     *
     * @param string $id ID of item
     *
     * @return mixed
     */
    public function getShareLink($id)
    {
        $path = '/' . $id;
        try {
            $onedriveconfig = get_option('_cloudfiles_onedrive_config');
            $client = Onedrive::client(
                $this->client_id,
                ['state' => $onedriveconfig['state']]
            );
            $shareId = $client->getDriveItemByPath($path)->createLink('embed', ['scope' => 'anonymous'])->shareId;
            return 'https://api.onedrive.com/v1.0/shares/' . $shareId . '/root/content';
        } catch (Exception $e) {
            error_log($e->getMessage());
            return false;
        }
    }

    public function get_shared_link($file, $settings=[]){
        $result = false;
        if(isset($file['path'])){
            $result = $this->getShareLink($file['path']);
        }
        return $result;
    }
}
