<?php

/**
 * Attachment filter
 *
 * @link       https://themeforest.net/user/nouthemes/portfolio
 * @since      1.0.0
 *
 * @package    Cloudfiles_Wordpress_Media_Library_Folders_Cloud
 * @subpackage Cloudfiles_Wordpress_Media_Library_Folders_Cloud/includes
 */

class Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Attachment_Filter {

	public static function init(){
        add_filter( 'wp_get_attachment_url', [__CLASS__, 'wp_get_attachment_url'], 1000000000, 2 );
        /*
        * Responsive Images WP 4.4
        */
        add_filter( 'wp_calculate_image_srcset', [__CLASS__, 'wp_calculate_image_srcset' ], 1000000000, 5 );
        add_filter( 'wp_calculate_image_srcset_meta', [__CLASS__, 'wp_calculate_image_srcset_meta' ], 1000000000, 4 );
        add_filter( 'wp_prepare_attachment_for_js', [__CLASS__, 'maybe_encode_wp_prepare_attachment_for_js'], 99, 3 );
    }

	/**
	 * Maybe encode URLs when outputting attachments in the media grid
	 *
	 * @param array      $response
	 * @param int|object $attachment
	 * @param array      $meta
	 *
	 * @return array
	 */
	public static function maybe_encode_wp_prepare_attachment_for_js( $response, $attachment, $meta ) {
		$provider_key = noucloudfiles_check_attachment_exists_meta($attachment->ID, 'noucloudfiles_provider_id');
        if(!$provider_key){
            return $response;
        }

		$sizes = '';

		if ( isset( $response['sizes'] ) && is_array( $response['sizes'] ) ) {
			$sizes = $response['sizes'];
		}
		
		if ($sizes) {
			foreach ( $sizes as $size => $value ) {
				if(isset($value['url'])){
                    $filename     = wp_basename( $value['url'] );
                    $provider_url = noucloudfiles_get_url_from_file_name( $attachment->ID, $provider_key, $filename );
        
                    if ( false === $provider_url || is_wp_error( $provider_url ) ) {
                        // Skip URLs not offloaded to cloud
                        continue;
                    }

					$response['sizes'][ $size ]['url'] = $provider_url;
				}
			}
        }
        
		$response['noucloudfiles_served'] = 'all';
		$s3_path = get_post_meta( $attachment->ID, 'noucloudfiles_file_url', true );
		if ( $s3_path != null ) {
			$response['noucloudfiles-cloud-class'] = "noucloudfiles-served-by-provider";
			$response['noucloudfiles_served'] = 1;
		}else{
			$response['noucloudfiles-cloud-class'] = "noucloudfiles-not-served";
		}

		return $response;
	}
	
	/**
	 * Get attachment url
	 *
	 * @param string $url
	 * @param int    $post_id
	 *
	 * @return bool|mixed|WP_Error
	 */
	public static function wp_get_attachment_url( $url, $post_id ) {
		$cloud_url = get_post_meta( $post_id, 'noucloudfiles_file_url', true);
		$new_url = false;
		
		if ( $cloud_url && $cloud_url != null ) {
			$new_url = $cloud_url;
		}

		if ( is_wp_error( $new_url ) || false === $new_url ) {
			return $url;
        }
        
		return apply_filters( 'noucloudfiles_get_attachment_url', $new_url, $post_id );

	}

	/**
	 * Replace local URLs with S3 ones for srcset image sources
	 *
	 * @param array  $sources
	 * @param array  $size_array
	 * @param string $image_src
	 * @param array  $image_meta
	 * @param int    $attachment_id
	 *
	 * @return array
	 */
	public static function wp_calculate_image_srcset( $sources, $size_array, $image_src, $image_meta, $attachment_id ) {
		$cloud_path = get_post_meta( $attachment_id, 'noucloudfiles_file_url', true);
		$new_url = false;
		
		if ( $cloud_path != null ) {
			$new_url = $cloud_path;
		}
		
		if ( is_wp_error( $new_url ) || false === $new_url ) {
			return $sources;
		}
				
		if ( ! is_array( $sources ) ) {
			// Sources corrupt
			return $sources;
		}
       
        $provider_key = noucloudfiles_check_attachment_exists_meta($attachment_id, 'noucloudfiles_provider_id');
        if($provider_key){
            foreach ( $sources as $width => $source ) {
                $filename     = wp_basename( $source['url'] );
                $provider_url = noucloudfiles_get_url_from_file_name( $attachment_id, $provider_key, $filename );
    
                if ( false === $provider_url || is_wp_error( $provider_url ) ) {
                    // Skip URLs not offloaded to cloud
                    continue;
                }
                
                $sources[ $width ]['url'] = $provider_url;
            }
        }

		return $sources;
	}

	/**
	 * Helper function to find size name from width and filename
	 *
	 * @param array  $sizes
	 * @param string $width
	 * @param string $filename
	 *
	 * @return null|string
	 */
	protected static function find_image_size_from_width( $sizes, $width, $filename ) {
		foreach ( $sizes as $name => $size ) {
			if ( $width === absint( $size['width'] ) && $size['file'] === $filename ) {
				return $name;
			}
		}

		return null;
	}

	/**
	 * Alter the image meta data to add srcset support for object versioned S3 URLs
	 *
	 * @param array  $image_meta
	 * @param array  $size_array
	 * @param string $image_src
	 * @param int    $attachment_id
	 *
	 * @return array
	 */
	public static function wp_calculate_image_srcset_meta( $image_meta, $size_array, $image_src, $attachment_id ) {
		$cloud_path = get_post_meta( $attachment_id, 'noucloudfiles_file_url', true);
		$new_url = false;
		
		if ( $cloud_path != null ) {
			$new_url = $cloud_path;
		}
		
		if ( is_wp_error( $new_url ) || false === $new_url ) {
			return $image_meta;
		}
		
		if ( empty( $image_meta['file'] ) ) {
			// Corrupt `_wp_attachment_metadata`
			return $image_meta;
		}

		$provider_key = noucloudfiles_check_attachment_exists_meta($attachment_id, 'noucloudfiles_provider_id');
        if(!$provider_key){
			// Attachment not uploaded to cloud, abort
			return $image_meta;
		}

		// Strip the meta file prefix so the just the filename will always match
		// the S3 URL regardless of different prefixes for the offloaded file.
        // Also ensure filename is encoded the same way as URL.
        $image_basename = wp_basename( $image_meta['file'] );
		$image_meta['file'] = rawurlencode( $image_basename );

		// Ensure each size filename is encoded the same way as URL.
		if ( ! empty( $image_meta['sizes'] ) ) {
			$image_meta['sizes'] = array_map( function ( $size ) {
				$size['file'] = rawurlencode( $size['file'] );

				return $size;
			}, $image_meta['sizes'] );
		}
        
		return $image_meta;
	}
}
