<?php
/**
 * Aws S3 Settings
 *
 * @link       https://themeforest.net/user/nouthemes/portfolio
 * @since      1.0.0
 *
 * @package    Cloudfiles_Wordpress_Media_Library_Folders_Cloud
 * @subpackage Cloudfiles_Wordpress_Media_Library_Folders_Cloud/admin
 */

defined( 'ABSPATH' ) || exit;

class Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Setting_Aws extends Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Settings_Page {

	/**
	 * Constructor.
	 */
    public $_array_regions = null;

	public function __construct() {
		$this->id    = CLOUDFILES_WORDPRESS_MEDIA_LIBRARY_FOLDERS_CLOUD_AWS;
		$this->label = esc_html__( 'Aws S3', 'cloudfiles-wordpress-media-library-folders-cloud' );
        $this->load_regions();

		parent::__construct();
    }

    public function load_regions() {
        $this->_array_regions = [
                'us-east-1'         => esc_html__( 'US East (N. Virginia)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'us-east-2'         => esc_html__( 'US East (Ohio)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'us-west-1'         => esc_html__( 'US West (N. California)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'us-west-2'         => esc_html__( 'US West (Oregon)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ca-central-1'      => esc_html__( 'Canada (Central)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-south-1'        => esc_html__( 'Asia Pacific (Mumbai)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-northeast-2'    => esc_html__( 'Asia Pacific (Seoul)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-southeast-1'    => esc_html__( 'Asia Pacific (Singapore)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-southeast-2'    => esc_html__( 'Asia Pacific (Sydney)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-northeast-1'    => esc_html__( 'Asia Pacific (Tokyo)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'eu-central-1'      => esc_html__( 'EU (Frankfurt)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'eu-west-1'         => esc_html__( 'EU (Ireland)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'eu-west-2'         => esc_html__( 'EU (London)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'sa-east-1'         => esc_html__( 'South America (São Paulo)', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'ap-east-1'         => esc_html__( 'Asia Pacific (Hong Kong)', 'cloudfiles-wordpress-media-library-folders-cloud' )
        ];
    }
    
    private function get_regions(){
        return array_keys($this->_array_regions);
    }

    private function s3_client( $region, $key, $Secret ) {

        $sdk = new Aws\Sdk( array(
            'region'      => $region,
            'version'     => 'latest',
            'credentials' => array(
                'key'    => $key,
                'secret' => $Secret,
            )
        ) );

        return $sdk->createS3();

    }

    private function get_bucket( $region, $key, $Secret ){

        $cache_key = '_cache_bucket_list_'.wp_hash($region.$key.$Secret, 'nonce');
        $cache_data = get_transient( $cache_key );
        
        if ( false === $cache_data ) {

            $regions    = $this->get_regions();
            $client     = $this->s3_client( $region, $key, $Secret );
            $buckets    = $client->listBuckets();
            $data       = [];

            foreach ( $buckets['Buckets'] as $bucket ) {

                try {
                    $result = $client->getBucketLocation(array(
                        'Bucket' => $bucket['Name'],
                    ));
                } catch ( Exception $e ) {
                    $result = false;
                }

                if($result){
                    if(in_array($result['LocationConstraint'], $regions)){
                        $data[$bucket['Name']] = $bucket['Name'];
                    }
                }
            }

            if(count($data) > 0){
                set_transient( $cache_key, $data, 60*15 );
            }

            return $data;
        }

        return $cache_data;
    }

	/**
	 * Get settings array.
	 *
	 * @return array
	 */
	public function get_settings() {

        $erasure_text = sprintf( '%s, <a href="%s" target="_blank">%s</a>', esc_html__('If you don\'t know where to search for your AWS S3 credentials', 'cloudfiles-wordpress-media-library-folders-cloud'), esc_url( 'https://aws.amazon.com/blogs/security/wheres-my-secret-access-key/' ), esc_html__('you can find them here', 'cloudfiles-wordpress-media-library-folders-cloud') );
        $value = get_option('noucloudfiles_enable_aws', 'yes');

        $options_default = [
            array(
                'title' => esc_html__( 'API CLient', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'type'  => 'title',
                'desc'  => '',
                'id'    => 'aws_section',
            ),
            array(
                'title' => esc_html__( 'Enable', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'type'  => 'checkbox',
                'value'  => $value,
                'id'    => 'noucloudfiles_enable_aws',
                'desc_tip' => false,
            ),
            array(
                'title' => esc_html__( 'Access key', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'type'  => 'password',
                'desc'  => $erasure_text,
                'id'    => 'aws_access_token'
            ),
            array(
                'title' => esc_html__( 'Secret access key', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'type'  => 'password',
                'desc'  => '',
                'id'    => 'aws_secret_access_token',
                'desc_tip' => false,
            ),
            array(
                'title' => esc_html__( 'Region', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                'type'  => 'select',
                'desc'  => '',
                'id'    => 'aws_region',
                'desc_tip' => false,
                'options' => $this->_array_regions
            ),
            array(
                'type' => 'sectionend',
                'id'   => 'aws_section',
            ),
        ];

        $option_settings = $options_default;

        $access_token           = get_option('aws_access_token');
		$secret_access_token    = get_option('aws_secret_access_token');
        $region                 = get_option('aws_region');
        if(!empty($access_token) && !empty($secret_access_token) && !empty($region)){
            $buckets = $this->get_bucket( $region, $access_token, $secret_access_token );
        
            $option_settings = [
                array(
                    'title' => esc_html__( 'API CLient', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'title',
                    'desc'  => '',
                    'id'    => 'aws_section',
                ),
				array(
					'title' => esc_html__( 'Enable', 'cloudfiles-wordpress-media-library-folders-cloud' ),
					'type'  => 'checkbox',
					'value'  => $value,
					'id'    => 'noucloudfiles_enable_aws',
					'desc_tip' => false,
				),
                array(
                    'title' => esc_html__( 'Access key', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'password',
                    'desc'  => $erasure_text,
                    'id'    => 'aws_access_token'
                ),
                array(
                    'title' => esc_html__( 'Secret access key', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'password',
                    'desc'  => '',
                    'id'    => 'aws_secret_access_token',
                    'desc_tip' => false,
                ),
                array(
                    'title' => esc_html__( 'Region', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'select',
                    'desc'  => '',
                    'id'    => 'aws_region',
                    'desc_tip' => false,
                    'options' => $this->_array_regions
                ),
                array(
                    'title' => esc_html__( 'Bucket', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'select',
                    'desc'  => '',
                    'id'    => 'aws_bucket',
                    'desc_tip' => false,
                    'options' => $buckets
                ),
                array(
                    'title' => esc_html__( 'Path', 'cloudfiles-wordpress-media-library-folders-cloud' ),
                    'type'  => 'text',
                    'desc'  => '',
                    'id'    => 'aws_path',
                    'desc_tip' => false,
                ),
                array(
                    'type' => 'sectionend',
                    'id'   => 'aws_section',
                ),
            ];
        }

		$settings = apply_filters(
			'noucloudfiles_aws_settings',
			$option_settings
		);

		return apply_filters( 'noucloudfiles_get_settings_' . $this->id, $settings );
	}

	/**
	 * Output the settings.
	 */
	public function output() {
		$settings = $this->get_settings();
		Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Settings::output_fields( $settings );
	}

	/**
	 * Save settings.
	 */
	public function save() {
		$settings = $this->get_settings();
		Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Admin_Settings::save_fields( $settings );

		$provider = new Cloudfiles_Wordpress_Media_Library_Folders_Cloud_Pro($this->id);
		if($provider->is_enable()){
            $provider->get_folder_cloud('', true, true);
            
            $Keyname = uniqid() . '_nou_cloudfiles_' . uniqid() . '.txt';
            $url = $provider->client->get_base_url( $Keyname );
            if(!empty($url)){
                update_option('aws_base_url', $url);
            }
		}
	}
}
